<?php
/**
 * Plugin Name: Simple Sitemap Generator
 * Plugin URI:  https://wordpress.org/google-custom-xml-sitemap
 * Description: Virtual XML and TXT sitemaps (posts, pages, tags, categories, Google News) with pagination support for large sites and optional gzip-compressed variants.
 * Version:     1.4.0
 * Author:      Local Seo Services
 * Author URI:  https://seoz.us/
 * Text Domain: simple-sitemap-generator
 * Domain Path: /languages
 * License:     GPLv2 or later
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'Simple_Sitemap_Generator' ) ) :

class Simple_Sitemap_Generator {

    const OPTION_KEY       = 'ssg_options';
    const URLS_PER_SITEMAP = 999; // Max URLs per sitemap piece (posts/tags)

    public function __construct() {
        // Settings
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_menu', array( $this, 'admin_menu' ) );

        // Rewrite & query vars
        add_action( 'init', array( $this, 'add_rewrite_rules' ) );
        add_filter( 'query_vars', array( $this, 'add_query_vars' ) );
        add_action( 'template_redirect', array( $this, 'template_redirect' ) );

        // robots.txt integration
        // NOTE: very high priority so we run LAST and can override other plugins.
        add_filter( 'robots_txt', array( $this, 'robots_txt' ), 999999, 2 );

        // Activation / deactivation
        register_activation_hook( __FILE__, array( 'Simple_Sitemap_Generator', 'activate' ) );
        register_deactivation_hook( __FILE__, array( 'Simple_Sitemap_Generator', 'deactivate' ) );
    }

    /**
     * Default options
     */
    public static function get_default_options() {
        return array(
            'enabled_post_types'    => array( 'post', 'page' ),
            'include_tags'          => 1,
            'include_categories'    => 1,
            'include_news'          => 1,
            'news_publication_name' => get_bloginfo( 'name' ),
            'exclude_categories'    => array(),
        );
    }

    /**
     * Get merged options
     */
    public static function get_options() {
        $defaults = self::get_default_options();
        $opts     = get_option( self::OPTION_KEY, array() );

        if ( ! is_array( $opts ) ) {
            $opts = array();
        }

        return wp_parse_args( $opts, $defaults );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting(
            'ssg_settings_group',
            self::OPTION_KEY,
            array( $this, 'sanitize_options' )
        );

        add_settings_section(
            'ssg_main_section',
            __( 'Simple Sitemap Generator Settings', 'simple-sitemap-generator' ),
            '__return_null',
            'ssg_settings_page'
        );

        add_settings_field(
            'enabled_post_types',
            __( 'Content Types Included in Sitemap', 'simple-sitemap-generator' ),
            array( $this, 'field_enabled_post_types' ),
            'ssg_settings_page',
            'ssg_main_section'
        );

        add_settings_field(
            'include_tags',
            __( 'Include Tags Sitemap', 'simple-sitemap-generator' ),
            array( $this, 'field_include_tags' ),
            'ssg_settings_page',
            'ssg_main_section'
        );

        add_settings_field(
            'include_categories',
            __( 'Include Categories Sitemap', 'simple-sitemap-generator' ),
            array( $this, 'field_include_categories' ),
            'ssg_settings_page',
            'ssg_main_section'
        );

        add_settings_field(
            'include_news',
            __( 'Enable Google News Sitemap', 'simple-sitemap-generator' ),
            array( $this, 'field_include_news' ),
            'ssg_settings_page',
            'ssg_main_section'
        );

        add_settings_field(
            'news_publication_name',
            __( 'Google News Publication Name', 'simple-sitemap-generator' ),
            array( $this, 'field_news_publication_name' ),
            'ssg_settings_page',
            'ssg_main_section'
        );

        add_settings_field(
            'exclude_categories',
            __( 'Excluded Categories', 'simple-sitemap-generator' ),
            array( $this, 'field_exclude_categories' ),
            'ssg_settings_page',
            'ssg_main_section'
        );
    }

    /**
     * Sanitize settings
     */
    public function sanitize_options( $input ) {
        $options = self::get_options();

        // Enabled post types
        $enabled_post_types = array();
        if ( isset( $input['enabled_post_types'] ) && is_array( $input['enabled_post_types'] ) ) {
            $allowed = array( 'post', 'page' );
            foreach ( $input['enabled_post_types'] as $pt ) {
                if ( in_array( $pt, $allowed, true ) ) {
                    $enabled_post_types[] = $pt;
                }
            }
        }
        $options['enabled_post_types'] = $enabled_post_types;

        // Tags
        $options['include_tags'] = ! empty( $input['include_tags'] ) ? 1 : 0;

        // Categories
        $options['include_categories'] = ! empty( $input['include_categories'] ) ? 1 : 0;

        // News
        $options['include_news'] = ! empty( $input['include_news'] ) ? 1 : 0;

        // Publication name
        if ( isset( $input['news_publication_name'] ) ) {
            $options['news_publication_name'] = sanitize_text_field( $input['news_publication_name'] );
        }

        // Excluded categories
        $exclude_cats = array();
        if ( isset( $input['exclude_categories'] ) && is_array( $input['exclude_categories'] ) ) {
            foreach ( $input['exclude_categories'] as $cat_id ) {
                $cat_id = absint( $cat_id );
                if ( $cat_id > 0 ) {
                    $exclude_cats[] = $cat_id;
                }
            }
        }
        $options['exclude_categories'] = $exclude_cats;

        flush_rewrite_rules();

        return $options;
    }

    /**
     * Settings fields
     */
    public function field_enabled_post_types() {
        $options = self::get_options();
        $values  = isset( $options['enabled_post_types'] ) ? (array) $options['enabled_post_types'] : array();
        ?>
        <label>
            <input type="checkbox"
                   name="<?php echo esc_attr( self::OPTION_KEY ); ?>[enabled_post_types][]"
                   value="post" <?php checked( in_array( 'post', $values, true ) ); ?> />
            <?php esc_html_e( 'Posts', 'simple-sitemap-generator' ); ?>
        </label><br />
        <label>
            <input type="checkbox"
                   name="<?php echo esc_attr( self::OPTION_KEY ); ?>[enabled_post_types][]"
                   value="page" <?php checked( in_array( 'page', $values, true ) ); ?> />
            <?php esc_html_e( 'Pages', 'simple-sitemap-generator' ); ?>
        </label>
        <?php
    }

    public function field_include_tags() {
        $options = self::get_options();
        ?>
        <label>
            <input type="checkbox"
                   name="<?php echo esc_attr( self::OPTION_KEY ); ?>[include_tags]"
                   value="1" <?php checked( ! empty( $options['include_tags'] ) ); ?> />
            <?php esc_html_e( 'Enable tags sitemaps (XML and TXT)', 'simple-sitemap-generator' ); ?>
        </label>
        <?php
    }

    public function field_include_categories() {
        $options = self::get_options();
        ?>
        <label>
            <input type="checkbox"
                   name="<?php echo esc_attr( self::OPTION_KEY ); ?>[include_categories]"
                   value="1" <?php checked( ! empty( $options['include_categories'] ) ); ?> />
            <?php esc_html_e( 'Enable categories sitemaps (XML and TXT)', 'simple-sitemap-generator' ); ?>
        </label>
        <?php
    }

    public function field_include_news() {
        $options = self::get_options();
        ?>
        <label>
            <input type="checkbox"
                   name="<?php echo esc_attr( self::OPTION_KEY ); ?>[include_news]"
                   value="1" <?php checked( ! empty( $options['include_news'] ) ); ?> />
            <?php esc_html_e( 'Enable /sitemap-news.xml and /sitemap-news.txt (Google News)', 'simple-sitemap-generator' ); ?>
        </label>
        <?php
    }

    public function field_news_publication_name() {
        $options = self::get_options();
        ?>
        <input type="text" class="regular-text"
               name="<?php echo esc_attr( self::OPTION_KEY ); ?>[news_publication_name]"
               value="<?php echo esc_attr( $options['news_publication_name'] ); ?>" />
        <p class="description">
            <?php esc_html_e( 'Publication name for Google News (e.g., your site brand name).', 'simple-sitemap-generator' ); ?>
        </p>
        <?php
    }

    public function field_exclude_categories() {
        $options    = self::get_options();
        $selected   = isset( $options['exclude_categories'] ) ? (array) $options['exclude_categories'] : array();
        $categories = get_categories( array( 'hide_empty' => false ) );
        ?>
        <p class="description">
            <?php esc_html_e( 'Selected categories will be excluded from posts and Google News sitemaps.', 'simple-sitemap-generator' ); ?>
        </p>
        <div style="max-height: 200px; overflow: auto; border: 1px solid #ccc; padding: 8px;">
            <?php foreach ( $categories as $cat ) : ?>
                <label style="display:block;">
                    <input type="checkbox"
                           name="<?php echo esc_attr( self::OPTION_KEY ); ?>[exclude_categories][]"
                           value="<?php echo esc_attr( $cat->term_id ); ?>"
                        <?php checked( in_array( $cat->term_id, $selected, true ) ); ?> />
                    <?php echo esc_html( $cat->name . ' (ID: ' . $cat->term_id . ')' ); ?>
                </label>
            <?php endforeach; ?>
        </div>
        <?php
    }

    /**
     * Admin menu
     */
    public function admin_menu() {
        add_options_page(
            __( 'Simple Sitemap Generator', 'simple-sitemap-generator' ),
            __( 'Simple Sitemap Generator', 'simple-sitemap-generator' ),
            'manage_options',
            'simple-sitemap-generator',
            array( $this, 'settings_page_html' )
        );
    }

    /**
     * Settings page HTML (with clickable links, including .xml.gz)
     */
    public function settings_page_html() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $base    = trailingslashit( home_url() );
        $options = self::get_options();

        // Prepare counts to know how many paginated sitemaps we have.
        $pages_count = 0;
        $posts_count = 0;
        $tags_count  = 0;

        if ( in_array( 'page', $options['enabled_post_types'], true ) ) {
            $pages_obj   = wp_count_posts( 'page' );
            $pages_count = isset( $pages_obj->publish ) ? (int) $pages_obj->publish : 0;
        }

        if ( in_array( 'post', $options['enabled_post_types'], true ) ) {
            $posts_count = $this->get_total_posts_for_sitemaps( $options['exclude_categories'] );
        }

        if ( ! empty( $options['include_tags'] ) ) {
            $tags_count = (int) wp_count_terms( array(
                'taxonomy'   => 'post_tag',
                'hide_empty' => true,
            ) );
        }

        $posts_pages = ( $posts_count > 0 ) ? (int) ceil( $posts_count / self::URLS_PER_SITEMAP ) : 0;
        $tags_pages  = ( $tags_count > 0 ) ? (int) ceil( $tags_count / self::URLS_PER_SITEMAP ) : 0;
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Simple Sitemap Generator', 'simple-sitemap-generator' ); ?></h1>

            <form method="post" action="options.php">
                <?php
                settings_fields( 'ssg_settings_group' );
                do_settings_sections( 'ssg_settings_page' );
                submit_button();
                ?>
            </form>

            <h2><?php esc_html_e( 'XML Sitemaps', 'simple-sitemap-generator' ); ?></h2>
            <ul>
                <li>
                    <a href="<?php echo esc_url( $base . 'sitemap-index.xml' ); ?>" target="_blank">
                        <?php echo esc_html( $base . 'sitemap-index.xml' ); ?>
                    </a>
                    <span class="description"> – XML sitemap index</span>
                </li>
                <?php if ( $pages_count > 0 && in_array( 'page', $options['enabled_post_types'], true ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-pages.xml' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-pages.xml' ); ?>
                        </a>
                    </li>
                <?php endif; ?>

                <?php if ( ! empty( $options['include_categories'] ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-categories.xml' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-categories.xml' ); ?>
                        </a>
                    </li>
                <?php endif; ?>

                <?php if ( $posts_pages > 0 && in_array( 'post', $options['enabled_post_types'], true ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-posts.xml' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-posts.xml' ); ?>
                        </a>
                        <span class="description">
                            (<?php esc_html_e( 'Additional parts: /sitemap-posts-2.xml, /sitemap-posts-3.xml, ...', 'simple-sitemap-generator' ); ?>)
                        </span>
                    </li>
                <?php endif; ?>

                <?php if ( $tags_pages > 0 && ! empty( $options['include_tags'] ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-tags.xml' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-tags.xml' ); ?>
                        </a>
                        <span class="description">
                            (<?php esc_html_e( 'Additional parts: /sitemap-tags-2.xml, /sitemap-tags-3.xml, ...', 'simple-sitemap-generator' ); ?>)
                        </span>
                    </li>
                <?php endif; ?>

                <?php if ( ! empty( $options['include_news'] ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-news.xml' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-news.xml' ); ?>
                        </a>
                    </li>
                <?php endif; ?>
            </ul>

            <h2><?php esc_html_e( 'TXT Sitemaps', 'simple-sitemap-generator' ); ?></h2>
            <ul>
                <li>
                    <a href="<?php echo esc_url( $base . 'sitemap-index.txt' ); ?>" target="_blank">
                        <?php echo esc_html( $base . 'sitemap-index.txt' ); ?>
                    </a>
                    <span class="description"> – TXT sitemap index</span>
                </li>
                <li>
                    <a href="<?php echo esc_url( $base . 'sitemap.txt' ); ?>" target="_blank">
                        <?php echo esc_html( $base . 'sitemap.txt' ); ?>
                    </a>
                    <span class="description"> – alias of sitemap-index.txt</span>
                </li>

                <?php if ( $pages_count > 0 && in_array( 'page', $options['enabled_post_types'], true ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-pages.txt' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-pages.txt' ); ?>
                        </a>
                    </li>
                <?php endif; ?>

                <?php if ( ! empty( $options['include_categories'] ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-categories.txt' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-categories.txt' ); ?>
                        </a>
                    </li>
                <?php endif; ?>

                <?php if ( $posts_pages > 0 && in_array( 'post', $options['enabled_post_types'], true ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-posts.txt' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-posts.txt' ); ?>
                        </a>
                        <span class="description">
                            (<?php esc_html_e( 'Additional parts: /sitemap-posts-2.txt, /sitemap-posts-3.txt, ...', 'simple-sitemap-generator' ); ?>)
                        </span>
                    </li>
                <?php endif; ?>

                <?php if ( $tags_pages > 0 && ! empty( $options['include_tags'] ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-tags.txt' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-tags.txt' ); ?>
                        </a>
                        <span class="description">
                            (<?php esc_html_e( 'Additional parts: /sitemap-tags-2.txt, /sitemap-tags-3.txt, ...', 'simple-sitemap-generator' ); ?>)
                        </span>
                    </li>
                <?php endif; ?>

                <?php if ( ! empty( $options['include_news'] ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-news.txt' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-news.txt' ); ?>
                        </a>
                    </li>
                <?php endif; ?>
            </ul>

            <h2><?php esc_html_e( 'GZipped XML Sitemaps (.xml.gz)', 'simple-sitemap-generator' ); ?></h2>
            <p class="description">
                <?php esc_html_e( 'These are gzip-compressed versions of the XML sitemaps. They contain exactly the same content as the .xml files, just compressed.', 'simple-sitemap-generator' ); ?>
            </p>
            <ul>
                <li>
                    <a href="<?php echo esc_url( $base . 'sitemap-index.xml.gz' ); ?>" target="_blank">
                        <?php echo esc_html( $base . 'sitemap-index.xml.gz' ); ?>
                    </a>
                </li>

                <?php if ( $pages_count > 0 && in_array( 'page', $options['enabled_post_types'], true ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-pages.xml.gz' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-pages.xml.gz' ); ?>
                        </a>
                    </li>
                <?php endif; ?>

                <?php if ( ! empty( $options['include_categories'] ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-categories.xml.gz' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-categories.xml.gz' ); ?>
                        </a>
                    </li>
                <?php endif; ?>

                <?php if ( $posts_pages > 0 && in_array( 'post', $options['enabled_post_types'], true ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-posts.xml.gz' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-posts.xml.gz' ); ?>
                        </a>
                        <span class="description">
                            (<?php esc_html_e( 'Additional parts: /sitemap-posts-2.xml.gz, /sitemap-posts-3.xml.gz, ...', 'simple-sitemap-generator' ); ?>)
                        </span>
                    </li>
                <?php endif; ?>

                <?php if ( $tags_pages > 0 && ! empty( $options['include_tags'] ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-tags.xml.gz' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-tags.xml.gz' ); ?>
                        </a>
                        <span class="description">
                            (<?php esc_html_e( 'Additional parts: /sitemap-tags-2.xml.gz, /sitemap-tags-3.xml.gz, ...', 'simple-sitemap-generator' ); ?>)
                        </span>
                    </li>
                <?php endif; ?>

                <?php if ( ! empty( $options['include_news'] ) ) : ?>
                    <li>
                        <a href="<?php echo esc_url( $base . 'sitemap-news.xml.gz' ); ?>" target="_blank">
                            <?php echo esc_html( $base . 'sitemap-news.xml.gz' ); ?>
                        </a>
                    </li>
                <?php endif; ?>
            </ul>

            <h2><?php esc_html_e( 'robots.txt', 'simple-sitemap-generator' ); ?></h2>
            <p>
                <a href="<?php echo esc_url( $base . 'robots.txt' ); ?>" target="_blank">
                    <?php echo esc_html( $base . 'robots.txt' ); ?>
                </a>
            </p>

            <p class="description">
                <?php esc_html_e( 'Make sure pretty permalinks are enabled in Settings → Permalinks.', 'simple-sitemap-generator' ); ?>
            </p>
        </div>
        <?php
    }

    /**
     * Rewrite rules
     */
    public function add_rewrite_rules() {
        // XML index
        add_rewrite_rule(
            '^sitemap-index\.xml$',
            'index.php?ssg_sitemap=index',
            'top'
        );

        // XML pages & categories (single)
        add_rewrite_rule(
            '^sitemap-pages\.xml$',
            'index.php?ssg_sitemap=pages',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-categories\.xml$',
            'index.php?ssg_sitemap=categories',
            'top'
        );

        // XML posts (paginated)
        add_rewrite_rule(
            '^sitemap-posts\.xml$',
            'index.php?ssg_sitemap=posts&ssg_page=1',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-posts-([0-9]+)\.xml$',
            'index.php?ssg_sitemap=posts&ssg_page=$matches[1]',
            'top'
        );

        // XML tags (paginated)
        add_rewrite_rule(
            '^sitemap-tags\.xml$',
            'index.php?ssg_sitemap=tags&ssg_page=1',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-tags-([0-9]+)\.xml$',
            'index.php?ssg_sitemap=tags&ssg_page=$matches[1]',
            'top'
        );

        // XML news
        add_rewrite_rule(
            '^sitemap-news\.xml$',
            'index.php?ssg_sitemap=news',
            'top'
        );

        // TXT index + alias
        add_rewrite_rule(
            '^sitemap-index\.txt$',
            'index.php?ssg_sitemap=txt_index',
            'top'
        );
        add_rewrite_rule(
            '^sitemap\.txt$',
            'index.php?ssg_sitemap=txt_index',
            'top'
        );

        // TXT pages & categories (single)
        add_rewrite_rule(
            '^sitemap-pages\.txt$',
            'index.php?ssg_sitemap=txt_pages',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-categories\.txt$',
            'index.php?ssg_sitemap=txt_categories',
            'top'
        );

        // TXT posts (paginated)
        add_rewrite_rule(
            '^sitemap-posts\.txt$',
            'index.php?ssg_sitemap=txt_posts&ssg_page=1',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-posts-([0-9]+)\.txt$',
            'index.php?ssg_sitemap=txt_posts&ssg_page=$matches[1]',
            'top'
        );

        // TXT tags (paginated)
        add_rewrite_rule(
            '^sitemap-tags\.txt$',
            'index.php?ssg_sitemap=txt_tags&ssg_page=1',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-tags-([0-9]+)\.txt$',
            'index.php?ssg_sitemap=txt_tags&ssg_page=$matches[1]',
            'top'
        );

        // TXT news
        add_rewrite_rule(
            '^sitemap-news\.txt$',
            'index.php?ssg_sitemap=txt_news',
            'top'
        );

        /*
         * GZIPPED XML SITEMAPS
         */
        add_rewrite_rule(
            '^sitemap-index\.xml\.gz$',
            'index.php?ssg_sitemap=index_gz',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-pages\.xml\.gz$',
            'index.php?ssg_sitemap=pages_gz',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-categories\.xml\.gz$',
            'index.php?ssg_sitemap=categories_gz',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-posts\.xml\.gz$',
            'index.php?ssg_sitemap=posts_gz&ssg_page=1',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-posts-([0-9]+)\.xml\.gz$',
            'index.php?ssg_sitemap=posts_gz&ssg_page=$matches[1]',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-tags\.xml\.gz$',
            'index.php?ssg_sitemap=tags_gz&ssg_page=1',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-tags-([0-9]+)\.xml\.gz$',
            'index.php?ssg_sitemap=tags_gz&ssg_page=$matches[1]',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-news\.xml\.gz$',
            'index.php?ssg_sitemap=news_gz',
            'top'
        );

        /*
         * GZIPPED TXT SITEMAPS
         * (supported as endpoints, but NOT listed in robots.txt)
         */
        add_rewrite_rule(
            '^sitemap-index\.txt\.gz$',
            'index.php?ssg_sitemap=txt_index_gz',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-pages\.txt\.gz$',
            'index.php?ssg_sitemap=txt_pages_gz',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-categories\.txt\.gz$',
            'index.php?ssg_sitemap=txt_categories_gz',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-posts\.txt\.gz$',
            'index.php?ssg_sitemap=txt_posts_gz&ssg_page=1',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-posts-([0-9]+)\.txt\.gz$',
            'index.php?ssg_sitemap=txt_posts_gz&ssg_page=$matches[1]',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-tags\.txt\.gz$',
            'index.php?ssg_sitemap=txt_tags_gz&ssg_page=1',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-tags-([0-9]+)\.txt\.gz$',
            'index.php?ssg_sitemap=txt_tags_gz&ssg_page=$matches[1]',
            'top'
        );
        add_rewrite_rule(
            '^sitemap-news\.txt\.gz$',
            'index.php?ssg_sitemap=txt_news_gz',
            'top'
        );
    }

    /**
     * Add query vars
     */
    public function add_query_vars( $vars ) {
        $vars[] = 'ssg_sitemap';
        $vars[] = 'ssg_page';
        return $vars;
    }

    /**
     * Handle virtual sitemap endpoints
     */
    public function template_redirect() {
        $type = get_query_var( 'ssg_sitemap' );
        if ( empty( $type ) ) {
            return;
        }

        $page = (int) get_query_var( 'ssg_page' );
        if ( $page < 1 ) {
            $page = 1;
        }

        switch ( $type ) {
            // XML
            case 'index':
                $this->output_sitemap_index_xml();
                break;
            case 'pages':
                $this->output_pages_sitemap_xml();
                break;
            case 'categories':
                $this->output_categories_sitemap_xml();
                break;
            case 'posts':
                $this->output_posts_sitemap_xml( $page );
                break;
            case 'tags':
                $this->output_tags_sitemap_xml( $page );
                break;
            case 'news':
                $this->output_news_sitemap_xml();
                break;

            // XML GZ
            case 'index_gz':
                $this->output_gz_from_url( '/sitemap-index.xml' );
                break;
            case 'pages_gz':
                $this->output_gz_from_url( '/sitemap-pages.xml' );
                break;
            case 'categories_gz':
                $this->output_gz_from_url( '/sitemap-categories.xml' );
                break;
            case 'posts_gz':
                $suffix = ( $page > 1 ) ? '-' . $page : '';
                $this->output_gz_from_url( '/sitemap-posts' . $suffix . '.xml' );
                break;
            case 'tags_gz':
                $suffix = ( $page > 1 ) ? '-' . $page : '';
                $this->output_gz_from_url( '/sitemap-tags' . $suffix . '.xml' );
                break;
            case 'news_gz':
                $this->output_gz_from_url( '/sitemap-news.xml' );
                break;

            // TXT
            case 'txt_index':
                $this->output_sitemap_index_txt();
                break;
            case 'txt_pages':
                $this->output_pages_sitemap_txt();
                break;
            case 'txt_categories':
                $this->output_categories_sitemap_txt();
                break;
            case 'txt_posts':
                $this->output_posts_sitemap_txt( $page );
                break;
            case 'txt_tags':
                $this->output_tags_sitemap_txt( $page );
                break;
            case 'txt_news':
                $this->output_news_sitemap_txt();
                break;

            // TXT GZ
            case 'txt_index_gz':
                $this->output_gz_from_url( '/sitemap-index.txt' );
                break;
            case 'txt_pages_gz':
                $this->output_gz_from_url( '/sitemap-pages.txt' );
                break;
            case 'txt_categories_gz':
                $this->output_gz_from_url( '/sitemap-categories.txt' );
                break;
            case 'txt_posts_gz':
                $suffix = ( $page > 1 ) ? '-' . $page : '';
                $this->output_gz_from_url( '/sitemap-posts' . $suffix . '.txt' );
                break;
            case 'txt_tags_gz':
                $suffix = ( $page > 1 ) ? '-' . $page : '';
                $this->output_gz_from_url( '/sitemap-tags' . $suffix . '.txt' );
                break;
            case 'txt_news_gz':
                $this->output_gz_from_url( '/sitemap-news.txt' );
                break;

            default:
                status_header( 404 );
                nocache_headers();
                exit;
        }

        exit;
    }

    /**
     * Output gzip-compressed content based on an existing sitemap URL.
     */
    protected function output_gz_from_url( $relative_url ) {
        $relative_url = '/' . ltrim( $relative_url, '/' );
        $url          = home_url( $relative_url );

        $response = wp_remote_get(
            $url,
            array(
                'timeout' => 30,
            )
        );

        if ( is_wp_error( $response ) || (int) wp_remote_retrieve_response_code( $response ) !== 200 ) {
            status_header( 500 );
            nocache_headers();
            exit;
        }

        $body   = wp_remote_retrieve_body( $response );
        $gzdata = gzencode( $body, 9 );

        header( 'Content-Type: application/x-gzip' );
        header( 'Content-Length: ' . strlen( $gzdata ) );

        echo $gzdata;
    }

    /* -------------------------------------------------------------------------
     * LOW-LEVEL HELPERS (SQL for posts & news, terms for tags)
     * ---------------------------------------------------------------------- */

    /**
     * Total posts for sitemap (respecting excluded categories)
     */
    protected function get_total_posts_for_sitemaps( $exclude_cats = array() ) {
        global $wpdb;

        $post_type = 'post';
        $status    = 'publish';

        $exclude_cats = array_map( 'absint', (array) $exclude_cats );
        $exclude_cats = array_filter( $exclude_cats );

        // No excluded categories: simple query
        if ( empty( $exclude_cats ) ) {
            $sql = $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->posts} p
                 WHERE p.post_type = %s AND p.post_status = %s",
                $post_type,
                $status
            );
            return (int) $wpdb->get_var( $sql );
        }

        $exclude_list = implode( ',', $exclude_cats );

        $sql = $wpdb->prepare(
            "
            SELECT COUNT(DISTINCT p.ID)
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->term_relationships} tr
                ON tr.object_id = p.ID
            LEFT JOIN {$wpdb->term_taxonomy} tt
                ON tt.term_taxonomy_id = tr.term_taxonomy_id
               AND tt.taxonomy = 'category'
            WHERE p.post_type = %s
              AND p.post_status = %s
              AND (tt.term_id IS NULL OR tt.term_id NOT IN ($exclude_list))
            ",
            $post_type,
            $status
        );

        return (int) $wpdb->get_var( $sql );
    }

    /**
     * Posts for a specific sitemap page (SQL, not WP_Query)
     */
    protected function get_posts_for_sitemap_page( $page, $exclude_cats = array() ) {
        global $wpdb;

        $page   = max( 1, (int) $page );
        $limit  = self::URLS_PER_SITEMAP;
        $offset = ( $page - 1 ) * $limit;

        $post_type = 'post';
        $status    = 'publish';

        $exclude_cats = array_map( 'absint', (array) $exclude_cats );
        $exclude_cats = array_filter( $exclude_cats );

        if ( empty( $exclude_cats ) ) {
            $sql = $wpdb->prepare(
                "
                SELECT p.ID, p.post_modified_gmt
                FROM {$wpdb->posts} p
                WHERE p.post_type  = %s
                  AND p.post_status = %s
                ORDER BY p.post_date DESC
                LIMIT %d OFFSET %d
                ",
                $post_type,
                $status,
                $limit,
                $offset
            );

            return $wpdb->get_results( $sql );
        }

        $exclude_list = implode( ',', $exclude_cats );

        $sql = $wpdb->prepare(
            "
            SELECT p.ID, p.post_modified_gmt
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->term_relationships} tr
                ON tr.object_id = p.ID
            LEFT JOIN {$wpdb->term_taxonomy} tt
                ON tt.term_taxonomy_id = tr.term_taxonomy_id
               AND tt.taxonomy = 'category'
            WHERE p.post_type  = %s
              AND p.post_status = %s
              AND (tt.term_id IS NULL OR tt.term_id NOT IN ($exclude_list))
            GROUP BY p.ID
            ORDER BY p.post_date DESC
            LIMIT %d OFFSET %d
            ",
            $post_type,
            $status,
            $limit,
            $offset
        );

        return $wpdb->get_results( $sql );
    }

    /**
     * Paginated tags
     */
    protected function get_tags_page_terms( $page ) {
        $page   = max( 1, (int) $page );
        $offset = ( $page - 1 ) * self::URLS_PER_SITEMAP;

        return get_terms( array(
            'taxonomy'   => 'post_tag',
            'hide_empty' => true,
            'number'     => self::URLS_PER_SITEMAP,
            'offset'     => $offset,
            'orderby'    => 'name',
            'order'      => 'ASC',
        ) );
    }

    /**
     * Recent posts for Google News (last 2 days) via SQL
     * Respects excluded categories.
     */
    protected function get_recent_news_posts( $exclude_cats = array() ) {
        global $wpdb;

        $post_type = 'post';
        $status    = 'publish';

        // We use local post_date, last 2 days starting from midnight
        $after_date = date( 'Y-m-d', strtotime( '-2 days' ) ) . ' 00:00:00';

        $exclude_cats = array_map( 'absint', (array) $exclude_cats );
        $exclude_cats = array_filter( $exclude_cats );

        if ( empty( $exclude_cats ) ) {
            $sql = $wpdb->prepare(
                "
                SELECT p.ID, p.post_date_gmt
                FROM {$wpdb->posts} p
                WHERE p.post_type  = %s
                  AND p.post_status = %s
                  AND p.post_date >= %s
                ORDER BY p.post_date DESC
                ",
                $post_type,
                $status,
                $after_date
            );

            return $wpdb->get_results( $sql );
        }

        $exclude_list = implode( ',', $exclude_cats );

        $sql = $wpdb->prepare(
            "
            SELECT DISTINCT p.ID, p.post_date_gmt
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->term_relationships} tr
                ON tr.object_id = p.ID
            LEFT JOIN {$wpdb->term_taxonomy} tt
                ON tt.term_taxonomy_id = tr.term_taxonomy_id
               AND tt.taxonomy = 'category'
            WHERE p.post_type  = %s
              AND p.post_status = %s
              AND p.post_date >= %s
              AND (tt.term_id IS NULL OR tt.term_id NOT IN ($exclude_list))
            ORDER BY p.post_date DESC
            ",
            $post_type,
            $status,
            $after_date
        );

        return $wpdb->get_results( $sql );
    }

    /* -------------------------------------------------------------------------
     * XML SITEMAPS
     * ---------------------------------------------------------------------- */

    /**
     * XML sitemap index: /sitemap-index.xml
     */
    protected function output_sitemap_index_xml() {
        $options = self::get_options();

        header( 'Content-Type: application/xml; charset=utf-8' );
        echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";

        $home = trailingslashit( home_url() );
        $now  = date( 'c' );

        // Counts
        $pages_count = 0;
        $posts_count = 0;
        $tags_count  = 0;

        if ( in_array( 'page', $options['enabled_post_types'], true ) ) {
            $pages_obj   = wp_count_posts( 'page' );
            $pages_count = isset( $pages_obj->publish ) ? (int) $pages_obj->publish : 0;
        }

        if ( in_array( 'post', $options['enabled_post_types'], true ) ) {
            $posts_count = $this->get_total_posts_for_sitemaps( $options['exclude_categories'] );
        }

        if ( ! empty( $options['include_tags'] ) ) {
            $tags_count = (int) wp_count_terms( array(
                'taxonomy'   => 'post_tag',
                'hide_empty' => true,
            ) );
        }

        $posts_pages = ( $posts_count > 0 ) ? (int) ceil( $posts_count / self::URLS_PER_SITEMAP ) : 0;
        $tags_pages  = ( $tags_count > 0 ) ? (int) ceil( $tags_count / self::URLS_PER_SITEMAP ) : 0;
        ?>
<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
    <?php if ( $pages_count > 0 && in_array( 'page', $options['enabled_post_types'], true ) ) : ?>
        <sitemap>
            <loc><?php echo esc_url( $home . 'sitemap-pages.xml' ); ?></loc>
            <lastmod><?php echo esc_html( $now ); ?></lastmod>
        </sitemap>
    <?php endif; ?>

    <?php if ( ! empty( $options['include_categories'] ) ) : ?>
        <sitemap>
            <loc><?php echo esc_url( $home . 'sitemap-categories.xml' ); ?></loc>
            <lastmod><?php echo esc_html( $now ); ?></lastmod>
        </sitemap>
    <?php endif; ?>

    <?php if ( $posts_pages > 0 && in_array( 'post', $options['enabled_post_types'], true ) ) :
        for ( $i = 1; $i <= $posts_pages; $i++ ) :
            $loc = ( $i === 1 )
                ? $home . 'sitemap-posts.xml'
                : $home . 'sitemap-posts-' . $i . '.xml';
            ?>
            <sitemap>
                <loc><?php echo esc_url( $loc ); ?></loc>
                <lastmod><?php echo esc_html( $now ); ?></lastmod>
            </sitemap>
        <?php endfor;
    endif; ?>

    <?php if ( $tags_pages > 0 && ! empty( $options['include_tags'] ) ) :
        for ( $i = 1; $i <= $tags_pages; $i++ ) :
            $loc = ( $i === 1 )
                ? $home . 'sitemap-tags.xml'
                : $home . 'sitemap-tags-' . $i . '.xml';
            ?>
            <sitemap>
                <loc><?php echo esc_url( $loc ); ?></loc>
                <lastmod><?php echo esc_html( $now ); ?></lastmod>
            </sitemap>
        <?php endfor;
    endif; ?>

    <?php if ( ! empty( $options['include_news'] ) ) : ?>
        <sitemap>
            <loc><?php echo esc_url( $home . 'sitemap-news.xml' ); ?></loc>
            <lastmod><?php echo esc_html( $now ); ?></lastmod>
        </sitemap>
    <?php endif; ?>
</sitemapindex>
        <?php
    }

    /**
     * XML: /sitemap-pages.xml
     */
    protected function output_pages_sitemap_xml() {
        $options = self::get_options();
        if ( ! in_array( 'page', $options['enabled_post_types'], true ) ) {
            status_header( 404 );
            exit;
        }

        $q = new WP_Query( array(
            'post_type'      => 'page',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'orderby'        => 'date',
            'order'          => 'DESC',
            'no_found_rows'  => true,
        ) );

        header( 'Content-Type: application/xml; charset=utf-8' );
        echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        ?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
    <?php if ( $q->have_posts() ) : ?>
        <?php while ( $q->have_posts() ) : $q->the_post(); ?>
            <url>
                <loc><?php the_permalink(); ?></loc>
                <lastmod><?php echo esc_html( get_post_modified_time( 'c', true ) ); ?></lastmod>
            </url>
        <?php endwhile; wp_reset_postdata(); ?>
    <?php endif; ?>
</urlset>
        <?php
    }

    /**
     * XML: /sitemap-categories.xml
     */
    protected function output_categories_sitemap_xml() {
        $options = self::get_options();
        if ( empty( $options['include_categories'] ) ) {
            status_header( 404 );
            exit;
        }

        $cats = get_terms( array(
            'taxonomy'   => 'category',
            'hide_empty' => true,
        ) );

        header( 'Content-Type: application/xml; charset=utf-8' );
        echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        ?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
    <?php foreach ( $cats as $cat ) :
        $link = get_term_link( $cat );
        if ( is_wp_error( $link ) ) {
            continue;
        }
        ?>
        <url>
            <loc><?php echo esc_url( $link ); ?></loc>
            <lastmod><?php echo esc_html( date( 'c' ) ); ?></lastmod>
        </url>
    <?php endforeach; ?>
</urlset>
        <?php
    }

    /**
     * XML: /sitemap-posts.xml and /sitemap-posts-X.xml
     */
    protected function output_posts_sitemap_xml( $page ) {
        $options = self::get_options();
        if ( ! in_array( 'post', $options['enabled_post_types'], true ) ) {
            status_header( 404 );
            exit;
        }

        $rows = $this->get_posts_for_sitemap_page( $page, $options['exclude_categories'] );

        header( 'Content-Type: application/xml; charset=utf-8' );
        echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        ?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
    <?php foreach ( $rows as $row ) :
        $post_id = (int) $row->ID;
        $url     = get_permalink( $post_id );
        if ( ! $url ) {
            continue;
        }
        $lastmod = get_post_modified_time( 'c', true, $post_id );
        ?>
        <url>
            <loc><?php echo esc_url( $url ); ?></loc>
            <lastmod><?php echo esc_html( $lastmod ); ?></lastmod>
        </url>
    <?php endforeach; ?>
</urlset>
        <?php
    }

    /**
     * XML: /sitemap-tags.xml and /sitemap-tags-X.xml
     */
    protected function output_tags_sitemap_xml( $page ) {
        $options = self::get_options();
        if ( empty( $options['include_tags'] ) ) {
            status_header( 404 );
            exit;
        }

        $tags = $this->get_tags_page_terms( $page );

        header( 'Content-Type: application/xml; charset=utf-8' );
        echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        ?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
    <?php foreach ( $tags as $tag ) :
        $link = get_term_link( $tag );
        if ( is_wp_error( $link ) ) {
            continue;
        }
        ?>
        <url>
            <loc><?php echo esc_url( $link ); ?></loc>
            <lastmod><?php echo esc_html( date( 'c' ) ); ?></lastmod>
        </url>
    <?php endforeach; ?>
</urlset>
        <?php
    }

    /**
     * XML: /sitemap-news.xml (last 2 days, via SQL)
     */
    protected function output_news_sitemap_xml() {
        $options = self::get_options();
        if ( empty( $options['include_news'] ) ) {
            status_header( 404 );
            exit;
        }

        $rows = $this->get_recent_news_posts( $options['exclude_categories'] );

        header( 'Content-Type: application/xml; charset=utf-8' );
        echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        ?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"
        xmlns:news="http://www.google.com/schemas/sitemap-news/0.9"
        xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">
    <?php
    $language         = get_bloginfo( 'language' );
    $publication_name = $options['news_publication_name'];

    foreach ( $rows as $row ) :
        $post_id   = (int) $row->ID;
        $permalink = get_permalink( $post_id );
        if ( ! $permalink ) {
            continue;
        }

        $pub_date = get_the_date( 'c', $post_id );
        $title    = get_the_title( $post_id );

        // Keywords from tags
        $tags     = get_the_terms( $post_id, 'post_tag' );
        $keywords = array();
        if ( ! empty( $tags ) && ! is_wp_error( $tags ) ) {
            foreach ( $tags as $tag ) {
                $keywords[] = $tag->name;
            }
        }
        $keywords_str = implode( ', ', $keywords );

        // Featured image
        $image_url = '';
        if ( has_post_thumbnail( $post_id ) ) {
            $img = wp_get_attachment_image_src( get_post_thumbnail_id( $post_id ), 'full' );
            if ( ! empty( $img[0] ) ) {
                $image_url = $img[0];
            }
        }
        ?>
        <url>
            <loc><?php echo esc_url( $permalink ); ?></loc>
            <news:news>
                <news:publication>
                    <news:name><?php echo esc_html( $publication_name ); ?></news:name>
                    <news:language><?php echo esc_html( $language ); ?></news:language>
                </news:publication>
                <news:publication_date><?php echo esc_html( $pub_date ); ?></news:publication_date>
                <news:title><?php echo esc_html( $title ); ?></news:title>
                <?php if ( ! empty( $keywords_str ) ) : ?>
                    <news:keywords><?php echo esc_html( $keywords_str ); ?></news:keywords>
                <?php endif; ?>
            </news:news>
            <?php if ( ! empty( $image_url ) ) : ?>
                <image:image>
                    <image:loc><?php echo esc_url( $image_url ); ?></image:loc>
                </image:image>
            <?php endif; ?>
        </url>
    <?php endforeach; ?>
</urlset>
        <?php
    }

    /* -------------------------------------------------------------------------
     * TXT SITEMAPS
     * ---------------------------------------------------------------------- */

    /**
     * TXT sitemap index: /sitemap-index.txt (and /sitemap.txt)
     */
    protected function output_sitemap_index_txt() {
        $options = self::get_options();
        $home    = trailingslashit( home_url() );

        $pages_count = 0;
        $posts_count = 0;
        $tags_count  = 0;

        if ( in_array( 'page', $options['enabled_post_types'], true ) ) {
            $pages_obj   = wp_count_posts( 'page' );
            $pages_count = isset( $pages_obj->publish ) ? (int) $pages_obj->publish : 0;
        }

        if ( in_array( 'post', $options['enabled_post_types'], true ) ) {
            $posts_count = $this->get_total_posts_for_sitemaps( $options['exclude_categories'] );
        }

        if ( ! empty( $options['include_tags'] ) ) {
            $tags_count = (int) wp_count_terms( array(
                'taxonomy'   => 'post_tag',
                'hide_empty' => true,
            ) );
        }

        $posts_pages = ( $posts_count > 0 ) ? (int) ceil( $posts_count / self::URLS_PER_SITEMAP ) : 0;
        $tags_pages  = ( $tags_count > 0 ) ? (int) ceil( $tags_count / self::URLS_PER_SITEMAP ) : 0;

        header( 'Content-Type: text/plain; charset=utf-8' );

        // pages.txt
        if ( $pages_count > 0 && in_array( 'page', $options['enabled_post_types'], true ) ) {
            echo esc_url_raw( $home . 'sitemap-pages.txt' ) . "\n";
        }

        // categories.txt
        if ( ! empty( $options['include_categories'] ) ) {
            echo esc_url_raw( $home . 'sitemap-categories.txt' ) . "\n";
        }

        // posts-X.txt
        if ( $posts_pages > 0 && in_array( 'post', $options['enabled_post_types'], true ) ) {
            for ( $i = 1; $i <= $posts_pages; $i++ ) {
                $path = ( $i === 1 ) ? 'sitemap-posts.txt' : 'sitemap-posts-' . $i . '.txt';
                echo esc_url_raw( $home . $path ) . "\n";
            }
        }

        // tags-X.txt
        if ( $tags_pages > 0 && ! empty( $options['include_tags'] ) ) {
            for ( $i = 1; $i <= $tags_pages; $i++ ) {
                $path = ( $i === 1 ) ? 'sitemap-tags.txt' : 'sitemap-tags-' . $i . '.txt';
                echo esc_url_raw( $home . $path ) . "\n";
            }
        }

        // news.txt
        if ( ! empty( $options['include_news'] ) ) {
            echo esc_url_raw( $home . 'sitemap-news.txt' ) . "\n";
        }
    }

    /**
     * TXT: /sitemap-pages.txt
     */
    protected function output_pages_sitemap_txt() {
        $options = self::get_options();
        if ( ! in_array( 'page', $options['enabled_post_types'], true ) ) {
            status_header( 404 );
            exit;
        }

        $q = new WP_Query( array(
            'post_type'      => 'page',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'orderby'        => 'date',
            'order'          => 'DESC',
            'no_found_rows'  => true,
        ) );

        header( 'Content-Type: text/plain; charset=utf-8' );

        if ( $q->have_posts() ) {
            while ( $q->have_posts() ) {
                $q->the_post();
                echo esc_url_raw( get_permalink() ) . "\n";
            }
            wp_reset_postdata();
        }
    }

    /**
     * TXT: /sitemap-categories.txt
     */
    protected function output_categories_sitemap_txt() {
        $options = self::get_options();
        if ( empty( $options['include_categories'] ) ) {
            status_header( 404 );
            exit;
        }

        $cats = get_terms( array(
            'taxonomy'   => 'category',
            'hide_empty' => true,
        ) );

        header( 'Content-Type: text/plain; charset=utf-8' );

        foreach ( $cats as $cat ) {
            $link = get_term_link( $cat );
            if ( is_wp_error( $link ) ) {
                continue;
            }
            echo esc_url_raw( $link ) . "\n";
        }
    }

    /**
     * TXT: /sitemap-posts.txt and /sitemap-posts-X.txt
     */
    protected function output_posts_sitemap_txt( $page ) {
        $options = self::get_options();
        if ( ! in_array( 'post', $options['enabled_post_types'], true ) ) {
            status_header( 404 );
            exit;
        }

        $rows = $this->get_posts_for_sitemap_page( $page, $options['exclude_categories'] );

        header( 'Content-Type: text/plain; charset=utf-8' );

        foreach ( $rows as $row ) {
            $url = get_permalink( (int) $row->ID );
            if ( ! $url ) {
                continue;
            }
            echo esc_url_raw( $url ) . "\n";
        }
    }

    /**
     * TXT: /sitemap-tags.txt and /sitemap-tags-X.txt
     */
    protected function output_tags_sitemap_txt( $page ) {
        $options = self::get_options();
        if ( empty( $options['include_tags'] ) ) {
            status_header( 404 );
            exit;
        }

        $tags = $this->get_tags_page_terms( $page );

        header( 'Content-Type: text/plain; charset=utf-8' );

        foreach ( $tags as $tag ) {
            $link = get_term_link( $tag );
            if ( is_wp_error( $link ) ) {
                continue;
            }
            echo esc_url_raw( $link ) . "\n";
        }
    }

    /**
     * TXT: /sitemap-news.txt (last 2 days via SQL)
     */
    protected function output_news_sitemap_txt() {
        $options = self::get_options();
        if ( empty( $options['include_news'] ) ) {
            status_header( 404 );
            exit;
        }

        $rows = $this->get_recent_news_posts( $options['exclude_categories'] );

        header( 'Content-Type: text/plain; charset=utf-8' );

        foreach ( $rows as $row ) {
            $url = get_permalink( (int) $row->ID );
            if ( ! $url ) {
                continue;
            }
            echo esc_url_raw( $url ) . "\n";
        }
    }

    /* -------------------------------------------------------------------------
     * robots.txt
     * ---------------------------------------------------------------------- */

    /**
     * Ensure robots.txt contains ONLY our sitemap references.
     * - Keep all existing User-agent / Disallow / Allow lines coming from other plugins.
     * - Remove ALL existing "Sitemap:" lines (even if minified on a single line).
     * - Append our XML + TXT sitemap URLs, including XML .gz variants.
     */
    public function robots_txt( $output, $public ) {
        if ( '0' === (string) $public ) {
            return $output;
        }

        $options = self::get_options();
        $base    = trailingslashit( home_url() );

        // Counts
        $pages_count = 0;
        $posts_count = 0;
        $tags_count  = 0;

        if ( in_array( 'page', $options['enabled_post_types'], true ) ) {
            $pages_obj   = wp_count_posts( 'page' );
            $pages_count = isset( $pages_obj->publish ) ? (int) $pages_obj->publish : 0;
        }

        if ( in_array( 'post', $options['enabled_post_types'], true ) ) {
            $posts_count = $this->get_total_posts_for_sitemaps( $options['exclude_categories'] );
        }

        if ( ! empty( $options['include_tags'] ) ) {
            $tags_count = (int) wp_count_terms( array(
                'taxonomy'   => 'post_tag',
                'hide_empty' => true,
            ) );
        }

        $posts_pages = ( $posts_count > 0 ) ? (int) ceil( $posts_count / self::URLS_PER_SITEMAP ) : 0;
        $tags_pages  = ( $tags_count > 0 ) ? (int) ceil( $tags_count / self::URLS_PER_SITEMAP ) : 0;

        $lines = array();

        // XML and TXT indexes
        $lines[] = 'Sitemap: ' . esc_url_raw( $base . 'sitemap-index.xml' );
        $lines[] = 'Sitemap: ' . esc_url_raw( $base . 'sitemap-index.xml.gz' ); // XML GZ index
        $lines[] = 'Sitemap: ' . esc_url_raw( $base . 'sitemap-index.txt' );
        $lines[] = 'Sitemap: ' . esc_url_raw( $base . 'sitemap.txt' );

        // XML pages/categories (+ gz)
        if ( $pages_count > 0 && in_array( 'page', $options['enabled_post_types'], true ) ) {
            $lines[] = 'Sitemap: ' . esc_url_raw( $base . 'sitemap-pages.xml' );
            $lines[] = 'Sitemap: ' . esc_url_raw( $base . 'sitemap-pages.xml.gz' );
        }
        if ( ! empty( $options['include_categories'] ) ) {
            $lines[] = 'Sitemap: ' . esc_url_raw( $base . 'sitemap-categories.xml' );
            $lines[] = 'Sitemap: ' . esc_url_raw( $base . 'sitemap-categories.xml.gz' );
        }

        // XML posts parts (+ gz)
        if ( $posts_pages > 0 && in_array( 'post', $options['enabled_post_types'], true ) ) {
            for ( $i = 1; $i <= $posts_pages; $i++ ) {
                if ( 1 === $i ) {
                    $xml_path    = 'sitemap-posts.xml';
                    $xml_gz_path = 'sitemap-posts.xml.gz';
                    $txt_path    = 'sitemap-posts.txt';
                } else {
                    $xml_path    = 'sitemap-posts-' . $i . '.xml';
                    $xml_gz_path = 'sitemap-posts-' . $i . '.xml.gz';
                    $txt_path    = 'sitemap-posts-' . $i . '.txt';
                }

                // XML
                $lines[] = 'Sitemap: ' . esc_url_raw( $base . $xml_path );
                // XML GZ
                $lines[] = 'Sitemap: ' . esc_url_raw( $base . $xml_gz_path );
                // TXT
                $lines[] = 'Sitemap: ' . esc_url_raw( $base . $txt_path );
            }
        }

        // XML tags parts (+ gz)
        if ( $tags_pages > 0 && ! empty( $options['include_tags'] ) ) {
            for ( $i = 1; $i <= $tags_pages; $i++ ) {
                if ( 1 === $i ) {
                    $xml_path    = 'sitemap-tags.xml';
                    $xml_gz_path = 'sitemap-tags.xml.gz';
                    $txt_path    = 'sitemap-tags.txt';
                } else {
                    $xml_path    = 'sitemap-tags-' . $i . '.xml';
                    $xml_gz_path = 'sitemap-tags-' . $i . '.xml.gz';
                    $txt_path    = 'sitemap-tags-' . $i . '.txt';
                }

                // XML
                $lines[] = 'Sitemap: ' . esc_url_raw( $base . $xml_path );
                // XML GZ
                $lines[] = 'Sitemap: ' . esc_url_raw( $base . $xml_gz_path );
                // TXT
                $lines[] = 'Sitemap: ' . esc_url_raw( $base . $txt_path );
            }
        }

        // XML + TXT news (+ xml.gz)
        if ( ! empty( $options['include_news'] ) ) {
            $lines[] = 'Sitemap: ' . esc_url_raw( $base . 'sitemap-news.xml' );
            $lines[] = 'Sitemap: ' . esc_url_raw( $base . 'sitemap-news.xml.gz' );
            $lines[] = 'Sitemap: ' . esc_url_raw( $base . 'sitemap-news.txt' );
        }

        // 1) Remove ALL existing "Sitemap:" directives from previous filters.
        //    This works even if everything is on a single line (minified).
        $clean_output = preg_replace( '/\s*Sitemap:\s*\S+/i', '', (string) $output );
        $clean_output = rtrim( $clean_output ) . "\n";

        // 2) Append our sitemap lines.
        $clean_output .= implode( "\n", $lines ) . "\n";

        return $clean_output;
    }

    /**
     * Activation
     */
    public static function activate() {
        $plugin = new self();
        $plugin->add_rewrite_rules();
        flush_rewrite_rules();

        if ( ! get_option( self::OPTION_KEY ) ) {
            add_option( self::OPTION_KEY, self::get_default_options() );
        }
    }

    /**
     * Deactivation
     */
    public static function deactivate() {
        flush_rewrite_rules();
    }
}

endif;

// Bootstrap the plugin
new Simple_Sitemap_Generator();
